{-# LANGUAGE OverloadedStrings #-}

module Main where

import System.Exit (exitFailure, exitSuccess)
import Test.HUnit hiding (Node)

import qualified Data.Text.Lazy as T
import qualified Data.Text.Lazy.Encoding as T
import qualified Data.Array.Unboxed as A
import qualified Data.ByteString.Lazy as B
import Nauty.Graph6.Internal
import Nauty.Internal.Parsing
import Control.Monad.Trans.State
import Data.Word

tests :: Test
tests = TestList                         
  [ TestLabel "numbers" $ TestCase
    ( do
      assertEqual ""
        (Right 2)
        (evalStateT parseNumber $ T.encodeUtf8 $ T.pack "A")
    )
  , TestLabel "parsing" $ TestCase
    ( do
      let testParsing g str = do
            assertEqual ("parse " ++ T.unpack str) ([Right g]) (parse str)
            assertEqual ("encode " ++ T.unpack str) str (encode g)
      let g0 = AdjacencyMatrix{numberOfVertices = 2, adjacency = A.array (0,0) [(0, 0)]}
          g1 = AdjacencyMatrix{numberOfVertices = 2, adjacency = A.array (0,0) [(0, 128)]}
          str0 = T.pack "A?"
          str1 = T.pack "A_"
      assertEqual ""
        ([Right g0])
        (parse str0)
      assertEqual ""
        ([Right g1])
        (parse str1)
      assertEqual ""
        (str0)
        (encode g0)
      assertEqual ""
        (str1)
        (encode g1)
      testParsing
        (AdjacencyMatrix{numberOfVertices = 1, adjacency = A.array (0,0) [(0,0)]})
        "@"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 6, adjacency = A.array (0,1) [(0, 0), (1,0)]})
        "E???"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 7, adjacency = A.array (0,2) [(0,0),(1,0),(2,0)]})
        "F????"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 8, adjacency = A.array (0,3) [(0,0),(1,0),(2,0), (3,0)]})
        "G?????"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 9, adjacency = A.array (0,4) [(0,0),(1,0),(2,0), (3,0), (4,0)]})
        "H??????"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 10, adjacency = A.array (0,5) [(0,0),(1,0),(2,0),(3,0),(4,0),(5,0)]})
        "I????????"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 11, adjacency = A.array (0,6) [(0,0),(1,0),(2,0),(3,0),(4,0),(5,0),(6,0)]})
        "J??????????"
      testParsing
        (AdjacencyMatrix{numberOfVertices = 11, adjacency = A.array (0,6) [(0,31),(1,185),(2,199),(3,14),(4,14),(5,7),(6,0)]})
        "JFzfFB_wF??"
      testParsing
        (AdjacencyMatrix {numberOfVertices = 9, adjacency = A.array (0,4) [(0,0),(1,1),(2,198),(3,136),(4,192)]})
        "H??FEaK"
    )
  , TestLabel "g = parse $ encode g" $ TestCase
    ( do 
      let g0 = fromEdgeList 4 [(0,1), (1,2), (2,3), (3,0)]
      assertEqual ""
        ("Cl")
        (encode g0)
      assertEqual ""
        ([Right g0])
        (parse $ encode g0)
      let g1 = fromEdgeList 10 [(0,1), (1,2), (2,3), (3,4), (4,5), (5,6), (6,7), (7,8), (8,9), (9,0)]
      assertEqual ""
        ("IhCGGC@_G")
        (encode g1)
      assertEqual ""
        ([Right g1])
        (parse $ encode g1)
    )
  , TestLabel "fromEdgeList" $ TestCase
    ( do 
      assertEqual ""
        (AdjacencyMatrix
          { numberOfVertices = 4
          , adjacency = A.array (0,0) [(0, 180)]})
        (fromEdgeList 4 [(0,1), (1,2), (2,3), (3,0)])
      assertEqual ""
        (4, [(0,1), (1,2), (0, 3), (2,3)])
        (toEdgeList $ fromEdgeList 4 [(0,1), (1,2), (2,3), (3,0)])
      assertEqual ""
        (AdjacencyMatrix
          { numberOfVertices = 10
          , adjacency = A.array (0,5) 
              [ (0, 164)
              , (1, 66)
              , (2, 8)
              , (3, 16)
              , (4, 24)
              , (5, 8)
              ]
          })
        (fromEdgeList 10 [(0,1), (1,2), (2,3), (3,4), (4,5), (5,6), (6,7), (7,8), (8,9), (9,0)])
      assertEqual ""
        (10, [(0,1), (1,2), (2,3), (3,4), (4,5), (5,6), (6,7), (7,8), (0, 9), (8,9)])
        (toEdgeList $ fromEdgeList 10 [(0,1), (1,2), (2,3), (3,4), (4,5), (5,6), (6,7), (7,8), (8,9), (9,0)])
    )
  , TestLabel "append6bits" $ TestCase
    ( do
      assertEqual ""
        (([164], 2))
        (append6Bits ([],0) (fromIntegral $ fromEnum 'h' :: Word8)  )
      assertEqual ""
        (([64,164], 4))
        (append6Bits ([164],2) (fromIntegral $ fromEnum 'C' :: Word8)  )
      assertEqual ""
        (([0,66,164], 6))
        (append6Bits ([64, 164], 4) (fromIntegral $ fromEnum 'G' :: Word8)  )
      assertEqual ""
        (([8,66,164], 0))
        (append6Bits ([0, 66, 164], 6) (fromIntegral $ fromEnum 'G' :: Word8)  )
      assertEqual ""
        (([16,8,66,164], 2))
        (append6Bits ([8, 66, 164], 0) (fromIntegral $ fromEnum 'C' :: Word8)  )
      assertEqual ""
        (([16,16,8,66,164], 4))
        (append6Bits ([16, 8, 66, 164], 2) (fromIntegral $ fromEnum '@' :: Word8)  )
      assertEqual ""
        (([0,24,16,8,66,164], 6))
        (append6Bits ([16, 16, 8, 66, 164], 4) (fromIntegral $ fromEnum '_' :: Word8)  )
      assertEqual ""
        (([8,24,16,8,66,164], 0))
        (append6Bits ([0,24,16,8,66,164], 6) (fromIntegral $ fromEnum 'G' :: Word8)  )
      assertEqual ""
        (Right $ [164,66,8,16,24,8])
        (fmap B.unpack $ evalStateT (parseVector 45) $ T.encodeUtf8 "hCGGC@_G")
      -- ???
      assertEqual ""
        (([0], 2))
        (append6Bits ([],0) (fromIntegral $ fromEnum '?' :: Word8)  )
      assertEqual ""
        (([0,0], 4))
        (append6Bits ([0],2) (fromIntegral $ fromEnum '?' :: Word8)  )
      assertEqual ""
        (([0,0,0], 6))
        (append6Bits ([0,0],4) (fromIntegral $ fromEnum '?' :: Word8)  )
      -- ??FEaK
      assertEqual ""
        (([192,1,0], 6))
        (append6Bits ([0,0],4) (fromIntegral $ fromEnum 'F' :: Word8)  )
      assertEqual ""
        (([198,1,0], 0))
        (append6Bits ([192,1,0],6) (fromIntegral $ fromEnum 'E' :: Word8)  )
      assertEqual ""
        (([136, 198,1,0], 2))
        (append6Bits ([198,1,0],0) (fromIntegral $ fromEnum 'a' :: Word8)  )
      assertEqual ""
        (([192,136, 198,1,0], 4))
        (append6Bits ([136,198,1,0],2) (fromIntegral $ fromEnum 'K' :: Word8)  )
    )
  ]

main :: IO ()
main = do 
  count <- runTestTT tests
  if errors count + failures count > 0 then exitFailure else exitSuccess
