{-|
Module      : Nauty.Internal
Description : Internal functions.
Copyright   : (c) Marcelo Garlet Milani, 2025
License     : GPL-3
Maintainer  : mgmilani@pm.me
Stability   : unstable

This module contains internal functions used by the "Nauty" module.
Except for test cases, you should not import this module.
-}

module Nauty.Internal where

import Data.Word
import qualified Data.Set       as Set
import qualified Data.Text.Lazy as T
import qualified Nauty.Digraph6 as D
import           Nauty.Internal.Utils
import           Nauty.Internal.Parsing
import qualified Nauty.Graph6   as G
import qualified Nauty.Sparse6  as S
import qualified Data.Array.Unboxed as A

-- |Whether the graph has undirected or directed edges.
data GraphType = Undirected | Directed deriving (Eq, Show)

-- |Format in used for encoding and decoding.
data Format = Graph6 | Sparse6 | Digraph6 | Incremental deriving (Eq, Show)

-- |A directed or undirected graph.
-- See nauty documentation for an explanation of the format used by the adjacency matrix.
data Graph = Graph
  { numberOfVertices :: Word64
  , graphType        :: GraphType
  , adjacencyList    :: A.UArray Word64 Word64 -- ^ List of edges. Computed from adjacency matrix in graph6 and digraph6 formats. Edges have the form (A[2i], A[2i + 1]), for every i from 0 to half the array size.
  , adjacencyMatrix  :: A.UArray Word64 Word8 -- ^ Adjacency matrix. Computed from edge list in sparse6 formats.
  , format           :: Format
  }
  deriving (Eq, Show)

-- |Parse a list of graphs encoded in graph6, sparse6 or digraph6 format. Each line encodes one graph.
-- Optionally, the first line might be a header of the form @>>graph6\<\<@, @>>sparse6\<\<@ or @>>digraph6\<\<@.
parse :: T.Text -> [Either T.Text Graph]
parse t = 
  let t' = ignoreHeader t
  in parse' Nothing (T.lines t')

-- |Parse a list of graphs in graph6, digraph6, sparse6 or incremental sparse6 format.
parse' :: (Maybe Graph) -- ^ Used in case the first graph of the list uses incremental sparse6 format.
       -> [T.Text]
       -> [Either T.Text Graph]
parse' _ [] = []
parse' g0 (g:gs) = 
  case getFormat g of
    Incremental -> case g0 of
            Nothing ->
               (Left $ "The graph [" <> g <> "] is in incremental sparse6 format but it is not preceded by a graph in graph6 or sparse6 format.")
               : parse' g0 (dropWhile (\h -> getFormat h == Incremental) gs)
            Just g0' ->
               case S.symmetricDifference 
                      (S.AdjacencyList
                        { S.numberOfVertices = g0'.numberOfVertices
                        , S.adjacency = g0'.adjacencyList
                        }) g of
              Left err -> [Left err]
              Right g' ->
                let g6 = (uncurry G.fromEdgeList) $ S.toEdgeList g'
                    g1 = Graph
                           { numberOfVertices = S.numberOfVertices g'
                           , adjacencyList = S.adjacency g'
                           , graphType = Undirected
                           , format = Incremental
                           , adjacencyMatrix = G.adjacency g6
                           }
                in Right g1 : parse' (Just g1) gs
             
    Sparse6 ->
      case S.graph g of
        Left err -> [Left err]
        Right g' ->
          let g6 = (uncurry G.fromEdgeList) $ S.toEdgeList g'
              g1 = Graph
                     { numberOfVertices = S.numberOfVertices g'
                     , adjacencyList = S.adjacency g'
                     , graphType = Undirected
                     , format = Sparse6
                     , adjacencyMatrix = G.adjacency g6
                     }
          in Right g1 : parse' (Just g1) gs
    Digraph6 ->
      case D.digraph g of
        Left err -> [Left err]
        Right g' ->
           let g1 = Graph
                      { numberOfVertices = D.numberOfVertices g'
                      , adjacencyList = 
                          let es = snd (D.toArcList g')
                              m = fromIntegral $ length es
                          in A.array (0, 2*m - 1) $ zip [0..] $ ungroupByTwo es
                      , graphType = Directed
                      , format = Digraph6
                      , adjacencyMatrix = D.adjacency g'
                      }
           in Right g1 : parse' Nothing gs
    Graph6 ->
      case G.graph g of
        Left err -> [Left err]
        Right g' ->
           let g1 = Graph
                      { numberOfVertices = G.numberOfVertices g'
                      , adjacencyList = 
                          let es = snd (G.toEdgeList g')
                              m = fromIntegral $ length es
                          in A.array (0, 2 * m - 1) $ zip [0..] $ ungroupByTwo es
                      , graphType = Undirected
                      , format = Graph6
                      , adjacencyMatrix = G.adjacency g'
                      }
           in Right g1 : parse' (Just g1) gs
 
-- |Format used for encoding the graph.
getFormat :: T.Text -> Format
getFormat str 
  | ":" `T.isPrefixOf` str = Sparse6
  | ";" `T.isPrefixOf` str = Incremental
  | "&" `T.isPrefixOf` str = Digraph6
  | otherwise = Graph6

-- | Encode a sequence of graphs using the given formats.
-- Graphs may be encoded with the incremental format, but only if the previous graph is encoded with sparse6.
encodeMany :: [Graph] -> T.Text
encodeMany gs = T.intercalate "\n" $ encodeMany' Nothing gs

-- | Encode a sequence of graphs using the given formats.
-- Graphs may be encoded with the incremental format, but only if the previous graph is encoded with sparse6.
encodeMany' :: Maybe Graph -- ^ Used as a predecessor in case the first graph of the list is to be encoded in incremental sparse6 format.
            -> [Graph]
            -> [T.Text]
encodeMany' _ [] = []
encodeMany' mg0 (g:gs) = 
  case g.format of
    Graph6 ->
      G.encode
        G.AdjacencyMatrix
          { G.numberOfVertices = g.numberOfVertices
          , G.adjacency  = g.adjacencyMatrix
          }
      : encodeMany' (Just g) gs
    Sparse6 ->
      S.encode
        S.AdjacencyList
          { S.numberOfVertices = g.numberOfVertices
          , S.adjacency = g.adjacencyList
          }
      : encodeMany' (Just g) gs
    Digraph6 ->
      D.encode
        D.AdjacencyMatrix
          { D.numberOfVertices = g.numberOfVertices
          , D.adjacency = g.adjacencyMatrix
          }
      : encodeMany' Nothing gs
    Incremental -> 
      case mg0 of
        Nothing ->
         S.encode
           S.AdjacencyList
             { S.numberOfVertices = g.numberOfVertices
             , S.adjacency = g.adjacencyList
             }
        Just g0 ->
          if g0.graphType == Undirected then
            let es0     = Set.fromList $ groupByTwo $ A.elems g0.adjacencyList
                es1     = Set.fromList $ groupByTwo $ A.elems g.adjacencyList
                symdiff = Set.toList $ Set.symmetricDifference es0 es1
            in S.encodeSymmetricDifference g.numberOfVertices symdiff
          else
            S.encode
              S.AdjacencyList
                { S.numberOfVertices = g.numberOfVertices
                , S.adjacency = g.adjacencyList
                }
      : encodeMany' (Just g) gs


-- | Encode a graph using the format given by the `format` field.
-- Graphs with the `Incremental` format are encoded using sparse6.
encode :: Graph -> T.Text
encode g =
  case g.format of
    Graph6 -> G.encode
                 G.AdjacencyMatrix
                   { G.numberOfVertices = g.numberOfVertices
                   , G.adjacency  = g.adjacencyMatrix
                   }
    Sparse6 -> S.encode
                S.AdjacencyList
                  { S.numberOfVertices = g.numberOfVertices
                  , S.adjacency = g.adjacencyList
                  }
    Digraph6 -> D.encode
                  D.AdjacencyMatrix
                    { D.numberOfVertices = g.numberOfVertices
                    , D.adjacency = g.adjacencyMatrix
                    }
    Incremental -> S.encode
                    S.AdjacencyList
                      { S.numberOfVertices = g.numberOfVertices
                      , S.adjacency = g.adjacencyList
                      }

-- | Whether the graph contains and edge between two vertices.
-- In digraph6 format, the edge is a directed edge from the first to the second vertex.
edgeExists :: Graph
           -> Word64 -- ^ From vertex in directed graphs.
           -> Word64 -- ^ To vertex in directed graphs.
           -> Bool
edgeExists g u v = 
  case g.format of
    Graph6  -> G.areAdjacent  
                 G.AdjacencyMatrix
                   { G.numberOfVertices = g.numberOfVertices
                   , G.adjacency  = g.adjacencyMatrix
                   }
                 u
                 v
    Incremental -> S.areAdjacent
                    S.AdjacencyList
                      { S.numberOfVertices = g.numberOfVertices
                      , S.adjacency = g.adjacencyList
                      }
                    u
                    v
    Sparse6  -> S.areAdjacent
                  S.AdjacencyList
                    { S.numberOfVertices = g.numberOfVertices
                    , S.adjacency = g.adjacencyList
                    }
                  u
                  v
    Digraph6 -> D.arcExists
                  D.AdjacencyMatrix
                    { D.numberOfVertices = g.numberOfVertices
                    , D.adjacency = g.adjacencyMatrix
                    }
                  u
                  v

-- |The number of vertices and the list of (potentially directed) edges of a graph.
toEdgeList :: Graph -> (Word64, [(Word64, Word64)])
toEdgeList graph = (graph.numberOfVertices, groupByTwo $ A.elems graph.adjacencyList)
