{-|
Module: Data.Type.Refine
Description: class of refinable types
Maintainer: Olaf Klinke
Stability: experimental

This module defines classes 
of refinable types and their 'Generic' representations. 

There are two ways to semi-automatically have a 'Refinable' instance for a given type @X@: 
Either the type is 'Generic', in which case a 'GRefine' instance 
can be derived for its 'Rep'resentation. 

@
instance Refinable X where
@

If the type is not 'Generic', you can use the implementation 

@
instance Refinable X where
    refine = refineDeMorgan
@

that applies the deMorgan laws until the predicate is known to either 
cover or delete the type. 
-}
{-# LANGUAGE GADTs, FlexibleContexts, FlexibleInstances, TypeOperators, DefaultSignatures #-}
module Data.Type.Refine (
    -- * Refinements
    Refinement,
    refinedTypeDef,
    runRefinement,
    -- * Refinable types
    Refinable(..),
    GRefine(..),
    refineDeMorgan,
    -- * Predicates (re-export)
    module Data.Type.Predicate
    ) where
import Data.Type.Predicate
import Type.Reflection (Typeable,SomeTypeRep(..),someTypeRep)
import Data.SubType (GSubType(..),idGSubType,mapGSubType,runGEmbedding)
import Data.Proxy (Proxy(..))
import GHC.Generics
import Topology.StoneSpace (StoneSpace,GStoneSpace(..))
import Data.Dynamic (Dynamic,fromDynamic)

-- * Refinements

-- | Sub-types are given as the predicate transformer duals 
-- of embeddings where the domain is existentially quantified. 
-- The domain is required to be refinable, too.
type Refinement = GSubType GStone GRefine

-- | Extract the type definition of the refined sub-type.
refinedTypeDef :: Refinement f -> SomeTypeRep
refinedTypeDef GEmpty = someTypeRep (Proxy :: Proxy (V1 ()))
refinedTypeDef (GEmbed h) = gDomain h

-- * Operations on refinements

unionRefinements :: Refinement f -> Refinement f -> Refinement f
unionRefinements GEmpty s = s
unionRefinements s GEmpty = s
unionRefinements (GEmbed (GStone f)) (GEmbed (GStone g)) = GEmbed (GStone (universalUnion f g))

-- | 'Refinement' is functorial 
mapRefinement :: GStone f g -> Refinement f -> Refinement g
mapRefinement = mapGSubType

-- | extract a marshalling function that yields 
-- the correct type as required by 'refinedTypeDef'.
toGDomain :: GRefine g => GStone g f -> Dynamic -> Maybe (g ())
toGDomain _ = fromDynamic -- repDynamic

runGRefinement :: GStoneSpace f => Refinement f -> Dynamic -> Maybe (f ())
runGRefinement = runGEmbedding toGDomain gSpec

-- | Since the refined sub-type is hidden inside an existential quantification, 
-- we can run the embedding only detour via 'Dynamic'. 
-- It is ensured that all 'Dynamic' values constructed from the type 
-- given by 'refinedTypeDef' will be mapped to a 'Just'. 
--
-- Executing a predicate transformer 'GStone' requires the target to be a 'StoneSpace'. 
runRefinement :: StoneSpace t => Refinement (Rep t) -> Dynamic -> Maybe t
runRefinement r = fmap to . runGRefinement r

-- * Refinable types

-- | Since 'Predicate' has kind @(Type -> Type) -> Type@ 
-- we have to wrap a monomorphic type in a 'K1'. 
-- This is in line with the usage of constants on Generic types. 
class Refinable c where
    refine :: Predicate (K1 R c) -> Refinement (K1 R c)
    default refine :: (Generic c, Rep c ~ f, GRefine f) => Predicate (K1 R c) -> Refinement (K1 R c)
    refine = mapRefinement (GStone mapConst) . grefine . mapConst

-- | Class of 'Rep'resentations of refinable types. 
-- The default implementation has inexhaustive pattern matches 
-- for any type with 'Predicate's that go beyond the Boolean connectives 
-- and should therefore be overridden. 
class Typeable f => GRefine f where
    grefine :: Predicate f -> Refinement f
    grefine = refineDeMorgan

instance GRefine U1 where
    grefine = refineDeMorgan

instance GRefine f => GRefine (Rec1 f) where
    grefine = mapRefinement (GStone mapRec) . grefine . mapRec

--  TODO: The sub-type forgets the meta-data. 
-- Find a way to push at least constructor and selector names 
-- down into the sub-type.
instance (GRefine f, Typeable i, Typeable m) => GRefine (M1 i m f) where
    grefine = mapRefinement (GStone mapMeta) . grefine . mapMeta

instance (GRefine f, GRefine g) => GRefine (f :+: g) where
    grefine p = unionRefinements
        (mapRefinement (GStone insL) (grefine (insL p)))
        (mapRefinement (GStone insR) (grefine (insR p)))

-- | 'first' lifted to refinements
refineFst :: GRefine g => Refinement f -> Refinement (f :*: g)
refineFst GEmpty = GEmpty
refineFst (GEmbed (GStone f)) = GEmbed (GStone (first f))

-- | 'second' lifted to refinements
refineSnd :: GRefine f => Refinement g -> Refinement (f :*: g)
refineSnd GEmpty = GEmpty
refineSnd (GEmbed (GStone g)) = GEmbed (GStone (second g))

-- | For types that have only the Boolean connectives as predicates, 
-- the only possible sub-types are the empty type and the entire type. 
-- Use this as default implementation of 'refine'.
refineDeMorgan :: GRefine f => Predicate f -> Refinement f
refineDeMorgan Wildcard = idGSubType
refineDeMorgan (p :\/ _) = refineDeMorgan p -- the type is join-prime
refineDeMorgan (p :/\ q) = case refineDeMorgan p of
    GEmpty -> GEmpty
    GEmbed _ -> refineDeMorgan q
refineDeMorgan (Neg p) = case p of
    Wildcard -> GEmpty
    Neg q    -> refineDeMorgan q -- double negation
    q :/\ r  -> refineDeMorgan (Neg q :\/ Neg r)
    q :\/ r  -> refineDeMorgan (Neg q :/\ Neg r)
-- Note: The warning about non-exhaustive pattern matches 
-- is deliberate - we have no way to constrain f to 
-- "all types where Predicate has only these constructors". 


instance (GRefine f, GRefine g) => GRefine (f :*: g) where
    grefine Wildcard = idGSubType
    grefine (Fst p)  = refineFst (grefine p)
    grefine (Snd p)  = refineSnd (grefine p)
    grefine (p :\/ q) = unionRefinements (grefine p) (grefine q)
    grefine (p :/\ q) = case grefine p of
        GEmpty -> GEmpty
        GEmbed e@(GStone h) -> mapRefinement e (grefine (h q))
    -- use deMorgan laws to refine a negation
    grefine (Neg p) = case p of
        Wildcard -> GEmpty
        Neg q    -> grefine q -- double negation
        q :/\ r  -> grefine (Neg q :\/ Neg r)
        q :\/ r  -> grefine (Neg q :/\ Neg r)
        Fst q    -> grefine (Fst (Neg q)) -- not . (\(x,_) -> p x) = \(x,_) -> (not.p) x
        Snd q    -> grefine (Snd (Neg q)) -- not . (\(_,x) -> p x) = \(_,x) -> (not.p) x

instance (Refinable c, Typeable c) => GRefine (K1 R c) where
    grefine = refine
