{-|
Module: Data.Type.Predicate
Description: statically analyzable predicates on generic types
Maintainer: Olaf Klinke
Stability: experimental

This module defines the GADT 'Predicate' 
and the category of predicate transformers.
-}
{-# LANGUAGE GADTs, PolyKinds, KindSignatures, TypeOperators, RankNTypes, ScopedTypeVariables #-}
module Data.Type.Predicate (
    -- * Predicates on polynomial types
    Predicate(..),
    eval,
    -- * Predicate transformers
    insL,
    insR,
    first,
    second,
    mapRec,
    mapMeta,
    mapConst,
    GStone(..),
    gDomain,
    -- * universal properties
    universalProd,
    universalUnion) where
import Type.Reflection (Typeable,SomeTypeRep,someTypeRep)
import Data.Kind (Type)
import Data.Proxy (Proxy(..))
import GHC.Generics
import Control.Applicative (liftA2)
import qualified Control.Category as Cat

-- | Predicates on Generic representations of polynomial types.
-- The only way to evaluate predicates on unions is to do 'Case' analysis.
-- All predicates on products must eventually extract the first or second component. 
data Predicate :: (Type -> Type) -> Type where
    Wildcard :: Predicate ty -- ^ the wildcard pattern
    Neg :: Predicate ty -> Predicate ty -- ^ logical negation
    (:\/) :: Predicate ty -> Predicate ty -> Predicate ty -- ^ logical disjunction
    (:/\) :: Predicate ty -> Predicate ty -> Predicate ty -- ^ logical conjunction
    Fst :: Predicate f -> Predicate (f :*: g)
    Snd :: Predicate g -> Predicate (f :*: g)
    Case :: Predicate f -> Predicate g -> Predicate (f :+: g)
    RecP :: Predicate f -> Predicate (Rec1 f)
    MetaP :: Predicate f -> Predicate (M1 i m f)
    ConstP :: (Generic c, Rep c ~ f) => Predicate f -> Predicate (K1 i c)
    
-- | evaluate a predicate
eval :: Predicate ty -> ty p -> Bool
eval Wildcard  _ = True
eval (Neg p) x = not (eval p x)
eval (p :\/ q) x = eval p x || eval q x
eval (p :/\ q) x = eval p x && eval q x
eval (Fst p) (x :*: _) = eval p x
eval (Snd p) (_ :*: x) = eval p x
eval (Case p _) (L1 x) = eval p x
eval (Case _ p) (R1 x) = eval p x
eval (RecP p) (Rec1 x) = eval p x
eval (MetaP p) (M1 x)  = eval p x
eval (ConstP p) (K1 x) = eval p (from x)

-- | universal property of the product @:*:@
-- and Stone dual of @(Control.Arrow.&&&)@. 
-- Notice that 'Fst' and 'Snd' are the Stone duals 
-- of 'fst' and 'snd', respectively. 
-- Accordingly, 
-- @'universalProd' 'Fst' 'Snd'@ 
-- is the identity.
universalProd :: 
    (Predicate a -> Predicate z) -> 
    (Predicate b -> Predicate z) -> 
    Predicate (a :*: b) -> Predicate z
universalProd f _ (Fst p) = f p
universalProd _ f (Snd p) = f p
universalProd _ _ Wildcard = Wildcard
universalProd f g (Neg p) = Neg (universalProd f g p)
universalProd f g (p :\/ q) = universalProd f g p :\/ universalProd f g q
universalProd f g (p :/\ q) = universalProd f g p :/\ universalProd f g q

-- | universal property of the union @:+:@
universalUnion :: 
    (Predicate z -> Predicate a) -> 
    (Predicate z -> Predicate b) -> 
    Predicate z -> Predicate (a :+: b)
universalUnion = liftA2 Case

-- * Predicate transformers 

-- | Predicate transformers form a cartesian category.
-- We can regard an element of @'GStone' ('Rep' x) ('Rep' y)@
-- as the Stone dual of a function @x -> y@.
-- Instead of the pair @(,)@ in this category 
-- @:*:@ plays the role of the cartesian product. 
newtype GStone f g = GStone (Predicate g -> Predicate f)
instance Cat.Category GStone where
    id = GStone id
    (GStone h) . (GStone g) = GStone (g.h)

-- | The monomorphic type representation 
-- of the domain @f@ of the morphism,
-- instantiated at parameter @()@.
gDomain :: forall f g. Typeable (f ()) => GStone f g -> SomeTypeRep
gDomain _ = someTypeRep (Proxy :: Proxy (f ()))

-- | 'insL' witnesses the inclusion of @f@ into @f :+: g@.
-- Stone dual of 'Left'.
insL :: Predicate (f :+: g) -> Predicate f
insL p = case p of
    Wildcard -> Wildcard
    Neg q -> Neg (insL q)
    q :\/ r -> insL q :\/ insL r
    q :/\ r -> insL q :/\ insL r
    Case q _ -> q

-- | 'insR' witnesses the inclusion of @g@ into @f :+: g@.
-- Stone dual of 'Right'.
insR :: Predicate (f :+: g) -> Predicate g
insR p = case p of
    Wildcard -> Wildcard
    Neg q -> Neg (insR q)
    q :\/ r -> insR q :\/ insR r
    q :/\ r -> insR q :/\ insR r
    Case _ q -> q

-- | inverse to 'RecP'
mapRec :: Predicate (Rec1 f) -> Predicate f
mapRec Wildcard = Wildcard
mapRec (Neg p) = Neg (mapRec p)
mapRec (p :\/ q) = mapRec p :\/ mapRec q
mapRec (p :/\ q) = mapRec p :/\ mapRec q
mapRec (RecP p) = p

-- | inverse to 'MetaP'
mapMeta :: Predicate (M1 i m f) -> Predicate f
mapMeta Wildcard = Wildcard
mapMeta (Neg p) = Neg (mapMeta p)
mapMeta (p :\/ q) = mapMeta p :\/ mapMeta q
mapMeta (p :/\ q) = mapMeta p :/\ mapMeta q
mapMeta (MetaP p) = p

-- | inverse to 'ConstP'
mapConst :: Generic c => Predicate (K1 i c) -> Predicate (Rep c)
mapConst Wildcard = Wildcard
mapConst (Neg p) = Neg (mapConst p)
mapConst (p :\/ q) = mapConst p :\/ mapConst q
mapConst (p :/\ q) = mapConst p :/\ mapConst q
mapConst (ConstP p) = p 

-- | Stone dual of @Control.Arrow.first@
first :: (Predicate f -> Predicate g) -> Predicate (f :*: h) -> Predicate (g :*: h)
first _ Wildcard = Wildcard
first f (Neg p) = Neg (first f p)
first f (p :/\ q) = first f p :/\ first f q
first f (p :\/ q) = first f p :\/ first f q
first f (Fst p) = Fst (f p)
first _ (Snd q) = Snd q

-- | Stone dual of @Control.Arrow.second@
second :: (Predicate f -> Predicate g) -> Predicate (h :*: f) -> Predicate (h :*: g)
second _ Wildcard = Wildcard
second f (Neg p) = Neg (second f p)
second f (p :/\ q) = second f p :/\ second f q
second f (p :\/ q) = second f p :\/ second f q
second _ (Fst p) = Fst p
second f (Snd q) = Snd (f q)

{-- TODO: does not compile
-- | Stone dual of @(Control.Arrow.***)@
pair :: (Predicate f -> Predicate f') -> 
    (Predicate g -> Predicate g') ->
    Predicate (f :*: f') -> Predicate (g :*: g')
pair _ _ Wildcard = Wildcard
pair f g (Neg p) = Neg (pair f g p)
pair f g (p :/\ q) = pair f g p :/\ pair f g q
pair f g (p :\/ q) = pair f g p :\/ pair f g q
pair f _ (Fst p) = Fst (f p)
pair _ g (Snd q) = Snd (g q)
--}
