{-|
Module: Data.SubType
Description: existentially quantified sub-type embeddings
Maintainer: Olaf Klinke
Stability: experimental

Functions with codomain `x`, regardless of their domain, 
can be encoded in the GADT 

@
data Codomain :: Type -> Type where
    Fun :: (y -> x) -> Codomain x
@

This module decorates this GADT with constraints on the domain `y` 
and generalizes functions to morphisms of arbitrary categories. 
-}
{-# LANGUAGE GADTs, KindSignatures, ConstraintKinds, PolyKinds, RankNTypes, ScopedTypeVariables #-}
module Data.SubType (SomeSubType(..),idSubType,runEmbedding,
    subDomain,runEmbeddingDynamic,
    GSubType(..),
    idGSubType,
    mapGSubType,
    runGEmbedding) where
import Data.Kind (Type,Constraint)
import Data.Proxy (Proxy(..))
import Data.Void (Void)
import Control.Category (Category,(<<<))
import qualified Control.Category as Cat
import Type.Reflection (Typeable,SomeTypeRep(..),someTypeRep)
import Data.Dynamic (Dynamic,fromDynamic)

-- * Sub-types in the category Hask

-- | Constrained sub-types. 
-- This data type records the embedding function 
-- but hides its domain in an existential type. 
-- This allows us to compute an embedding at run-time. 
-- As an extra parameter, we pass a type class so that only 
-- those sub-types are representable that are member of this class.
data SomeSubType :: (Type -> Constraint) -> Type -> Type where
    EmptySubType :: SomeSubType con t
    Embed :: con x => (x -> y) -> SomeSubType con y
-- TODO: any Category would do here.

domain :: forall cat a b. Typeable a => cat a b -> SomeTypeRep
domain _ = someTypeRep (Proxy :: Proxy a)

-- | The representation of the type of the domain of the sub-type embedding.
subDomain :: SomeSubType Typeable x -> SomeTypeRep
subDomain EmptySubType = someTypeRep (Proxy :: Proxy Void)
subDomain (Embed f) = domain f

-- | post-compose embeddings with functions. 
-- This allows us to nest embeddings. 
instance Functor (SomeSubType con) where
    fmap _ EmptySubType = EmptySubType
    fmap g (Embed f) = Embed (g.f)

-- | embed a type into itself via the identity function.
idSubType :: con t => SomeSubType con t
idSubType = Embed id

-- | In order to run an embedding with unknown domain, 
-- we need a function that can convert a single universal type @rep@ 
-- to any sub-type. 
-- 'Nothing' is returned if the @rep@ value does not represent 
-- an element of the sub-type, in particular if the sub-type is empty. 
runEmbedding :: (forall x. con x => rep -> Maybe x) -> 
    SomeSubType con y -> 
    rep -> Maybe y
runEmbedding maybeFrom (Embed f) = fmap f . maybeFrom
runEmbedding _ EmptySubType = const Nothing

-- | It is guaranteed that for every @f :: y -> x@ with @Typeable y@,
-- 
-- @
-- runEmbeddingDynamic (Embed f) (toDyn y) = Just (f y)
-- @
runEmbeddingDynamic :: SomeSubType Typeable x -> Dynamic -> Maybe x
runEmbeddingDynamic = runEmbedding fromDynamic

-- * Sub-types in arbitrary categories

-- | Poly-kinded version of 'SomeSubType' 
-- generalized to arbitrary categories.
data GSubType :: (k -> k -> Type) -> (k -> Constraint) -> k -> Type where
    GEmpty :: GSubType cat con x
    GEmbed :: con y => cat y x -> GSubType cat con x
    -- TODO: an unconstrained Gid would be useful, 
    -- so that types that do not satisfy @con@ can have at least the identity sub-type.

-- | the identiy morphism
idGSubType :: (Category cat, con x) => GSubType cat con x
idGSubType = GEmbed (Cat.id)

-- | @'GSubType' cat con@ is functorial on @cat@.
mapGSubType :: Category cat => cat x y -> GSubType cat con x -> GSubType cat con y
mapGSubType h (GEmbed f) = GEmbed (h <<< f)
mapGSubType _ GEmpty = GEmpty

-- | generic version of 'runEmbedding'. 
runGEmbedding :: (forall g. con g => cat g f -> Dynamic -> Maybe (g ())) -- ^ dynamically convert to the sub-type
    -> (forall s . cat s f -> s () -> f ()) -- ^ run the @cat@ morphism
    -> GSubType cat con f -- ^ the sub-type embedding
    -> Dynamic            -- ^ dynamic representation of the argument
    -> Maybe (f ())
runGEmbedding maybeFrom runCat (GEmbed h) = fmap (runCat h) . maybeFrom h
runGEmbedding _ _ GEmpty = const Nothing
-- TODO: generalizing Dynamic to @rep@ does not work. Why?
