{-|
Module: Data.Searchable 
Description: exhaustively searchable subsets
Maintainer: Olaf Klinke
Stability: experimental

There exist two packages for exhaustive search: 
[infinite-search](https://hackage.haskell.org/package/infinite-search)
and the 
[Select](https://hackage.haskell.org/package/transformers-0.6.1.2/docs/Control-Monad-Trans-Select.html)
monad transformer. 

This module extends the former (which has a much smaller dependency footprint) 
with the empty subset. 
The resulting monad 'K' does not only support unions 
but also intersections. 
-} 
module Data.Searchable (
    -- * Data types
    K(..),
    -- Set Operations
    restrict,
    intersection,
    -- * Queries
    exists,forevery,member,
    -- * Construction
    list2K) where
import Control.Applicative (Alternative(..))
import Control.Monad (MonadPlus(..),ap)
import Data.Set (Set)
import qualified Data.Set
import Data.List (intercalate)

-- | If sub-sets of a type admit 
-- continuous universal quantification ∀, 
-- then the empty sub-set is an isolated element: 
-- Indeed, for no other set should @∀ ('const' 'False')@ be true. 
-- 
-- Therefore it is sound to have a separate constructor for the empty set.
data K a = Emptyset 
    | Nonempty ((a -> Bool) -> a) -- ^ selection function

-- | The non-empty part of 'K'.
data S a = Finder {find :: (a -> Bool) -> a}

-- | existential quantification
exists :: K a -> (a -> Bool) -> Bool
exists Emptyset _ = False
exists (Nonempty f) p = p (f p)

-- | universal quantification
forevery :: K a -> (a -> Bool) -> Bool
forevery k = \p -> not (exists k (not.p))

s2k :: S a -> K a
s2k = Nonempty . find

union :: K (K a) -> K a
union Emptyset = Emptyset -- ⋃ ∅ = ∅
union (Nonempty ff) = case ff (const True) of
    Emptyset   -> Emptyset -- ⋃ {∅} = ∅
    Nonempty f -> Nonempty (\p -> case ff (\k -> exists k p) of
        Emptyset       -> f     p
        Nonempty find' -> find' p)

instance Functor K where
    fmap _ Emptyset = Emptyset
    fmap h (Nonempty f) = Nonempty (\p -> (h.f) (p.h))
instance Functor S where
    fmap f s = Finder (\p -> (f.(find s)) (p.f))
instance Monad K where
    return = pure
    s >>= k = union (fmap k s)
instance Monad S where
    return = pure
    (Finder fnd) >>= k = Finder (\p -> let 
        f = flip (find.k) p
        in f (fnd (p.f)))
instance Alternative K where
    empty = Emptyset
    Emptyset <|> k = k
    k@(Nonempty _) <|> Emptyset = k
    (Nonempty f) <|> (Nonempty g) = Nonempty (\p -> let a = f p in if p a then a else g p)
instance MonadPlus K where
    mzero = Emptyset
    mplus = (<|>)
instance Applicative K where
    pure = Nonempty . const
    (<*>) = ap
instance Applicative S where
    pure = Finder . const
    (<*>) = ap
instance (Ord a,Show a) => Show (K a) where
    show k = '{':(intercalate "," . map show . Data.Set.toList . k2Set $ k)++"}"
instance (Ord a,Show a) => Show (S a) where
    show = show.s2k
instance (Eq a) => Eq (K a) where
    (==) = egliMilner (==)

-- | membership is decidable for discrete spaces
member :: Eq a => a -> K a -> Bool
member = flip contains

-- | In a discrete space, compact sets are clopen. 
contains :: Eq a => K a -> a -> Bool
k `contains` x = exists k (x==)

-- | Intersect a compact with a clopen
restrict :: K a -> (a -> Bool) -> K a -- intersect a compact with a clopen.
restrict Emptyset _ = Emptyset
restrict k@(Nonempty f) p = if forevery k (not.p)
    then Emptyset
    else Nonempty (\q -> f (\a -> p a && q a))

-- | If every compact set is clopen, then the space is discrete. 
-- Indeed, discreteness means singletons are open. 
-- Singletons are always compact. 
intersection :: Eq a => K a -> K a -> K a
intersection Emptyset _ = Emptyset
intersection _ Emptyset = Emptyset
intersection k1 k2 = restrict k1 (contains k2)

-- | if k is compact, then so is {x} ∪ k
consK :: a -> K a -> K a
consK x Emptyset = return x
consK x (Nonempty f) = Nonempty (\p -> if p x then x else f p)

-- | convert lists
-- 
-- >>> list2K [4,2,2,2,3 :: Integer]
-- {2,3,4}
list2K :: Foldable f => f a -> K a
list2K = foldr consK Emptyset

k2Set :: Ord a => K a -> Set a
k2Set k = go Data.Set.empty k where
    go s Emptyset = s
    go s (Nonempty f) = let x = f (flip Data.Set.notMember s) in
        if x == f (x==)
            then let s' = Data.Set.insert x s in go 
                s'
                (restrict (Nonempty f) (flip Data.Set.notMember s'))
            else s

-- * Relation liftings

-- | Smyth relation lifting
smyth :: (a -> b -> Bool) -> K a -> K b -> Bool
smyth r xs ys = forevery ys ((exists xs).(flip r))
-- | Hoare relation lifting
hoare :: (a -> b -> Bool) -> K a -> K b -> Bool
hoare r xs ys = forevery xs ((exists ys).r)
-- | Egli-Milner relation lifting
egliMilner :: (a -> b -> Bool) -> K a -> K b -> Bool
egliMilner r xs ys = hoare r xs ys && smyth r xs ys
